<?php

/*
 * This file is part of Contao.
 *
 * Copyright (c) 2005-2017 Leo Feyer
 *
 * @license LGPL-3.0+
 */

namespace Contao\ManagerPlugin\Bundle;

use Contao\ManagerPlugin\Bundle\Config\ConfigInterface;
use Contao\ManagerPlugin\Bundle\Config\ConfigResolverFactory;
use Contao\ManagerPlugin\Bundle\Parser\ParserInterface;
use Contao\ManagerPlugin\PluginLoader;
use Symfony\Component\Filesystem\Filesystem;

class BundleLoader
{
    /**
     * @var PluginLoader
     */
    private $pluginLoader;

    /**
     * @var ConfigResolverFactory
     */
    private $resolverFactory;

    /**
     * @var ParserInterface
     */
    private $parser;

    /**
     * @var Filesystem
     */
    private $filesystem;

    /**
     * @param PluginLoader          $pluginLoader
     * @param ConfigResolverFactory $resolverFactory
     * @param ParserInterface       $parser
     * @param Filesystem            $filesystem
     */
    public function __construct(PluginLoader $pluginLoader, ConfigResolverFactory $resolverFactory, ParserInterface $parser, Filesystem $filesystem = null)
    {
        $this->pluginLoader = $pluginLoader;
        $this->resolverFactory = $resolverFactory;
        $this->parser = $parser;
        $this->filesystem = $filesystem;

        if (null === $this->filesystem) {
            $this->filesystem = new Filesystem();
        }
    }

    /**
     * Returns an ordered bundles map.
     *
     * @param bool        $development
     * @param string|null $cacheFile
     *
     * @return ConfigInterface[]
     */
    public function getBundleConfigs($development, $cacheFile = null)
    {
        if (null !== $cacheFile) {
            return $this->loadFromCache($development, $cacheFile);
        }

        return $this->loadFromPlugins($development, $cacheFile);
    }

    /**
     * Loads the bundles map from cache.
     *
     * @param bool        $development
     * @param string|null $cacheFile
     *
     * @return ConfigInterface[]
     */
    private function loadFromCache($development, $cacheFile)
    {
        $bundleConfigs = is_file($cacheFile) ? include $cacheFile : null;

        if (!\is_array($bundleConfigs) || 0 === \count($bundleConfigs)) {
            $bundleConfigs = $this->loadFromPlugins($development, $cacheFile);
        }

        return $bundleConfigs;
    }

    /**
     * Generates the bundles map.
     *
     * @param bool        $development
     * @param string|null $cacheFile
     *
     * @return ConfigInterface[]
     */
    private function loadFromPlugins($development, $cacheFile)
    {
        $resolver = $this->resolverFactory->create();

        /** @var BundlePluginInterface[] $plugins */
        $plugins = $this->pluginLoader->getInstancesOf(PluginLoader::BUNDLE_PLUGINS);

        foreach ($plugins as $plugin) {
            foreach ($plugin->getBundles($this->parser) as $config) {
                $resolver->add($config);
            }
        }

        $bundleConfigs = $resolver->getBundleConfigs($development);

        if (null !== $cacheFile) {
            $this->filesystem->dumpFile($cacheFile, sprintf('<?php return %s;', var_export($bundleConfigs, true)));
        }

        return $bundleConfigs;
    }
}
